/*************************************************************************
 * The contents of this file are subject to the MYRICOM MX AND GM-2      *
 * MAPPING SOFTWARE AND DOCUMENTATION LICENSE (the "License"); User may  *
 * not use this file except in compliance with the License.  The full    *
 * text of the License can found in mapper directory in LICENSE.TXT      *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

#include <ctype.h>
#include "lx.h"
#include "lx_route_file.h"

#define LX_ROUTE_FILE_MAX_LINE 200
#define LX_ROUTE_FILE_MAX_PORTS 2

#define lx_route_file_error(c,e){printf ("%d: ", (c));printf e; printf ("\n"); goto error;}

static char*lx_route_file_next_line (FILE*fp, char*line, int max_line, int*line_count)
{
  char*s;
  insist (fp && line && max_line > 0 && line_count);

  while ((s = fgets (line, max_line, fp)))
  {
    (*line_count)++;
    while (isspace ((int)*s))
      s++;
    if (*s) return s;
  }
  except: return 0;
}


int lx_route_file_read_route (char*route_string, lx_route_t*route)
{
  int hop;
  char*s;
  
  insist (route_string && route);
  route->length = 0;
  
  if (strstr (route_string, ".")) return -1;
  
  for (s = strtok (route_string, ","); s; s = strtok (0, ","))
  {
    if (route->length >= LX_EFFECTIVE_ROUTE_SIZE ||
	(hop = atoi (s)) < 1 - LX_XBAR_SIZE || hop >= LX_XBAR_SIZE)
      return 0;
    route->hops [route->length++] = hop;
  }
  return 1;
  except: return 0;
}

int lx_route_file_write_header (FILE*fp, lx_map_t*m, lx_node_t*me, int num_ports, int num_routes)
{
  insist (fp && m);
  insist (num_routes > 0 && num_ports > 0 && num_ports <= LX_ROUTE_FILE_MAX_PORTS);
  
  fprintf (fp, "h%d %d %d %d\n", me->index, m->num_hosts, num_ports, num_routes);
  return 1;
  except: return 0;
}


int lx_route_file_write_route (FILE*fp, int iport, int host, int index, int oiport, lx_route_t*route)
{
  insist (fp);

  fprintf (fp, "%d h%d %d %d %s\n", iport, host, index, oiport, route ? lx_print_route (route->hops, route->length) : ".");
  return 1;
  except: return 0;
}


int lx_route_file_write_routes (FILE*fp, lx_map_t*m, lx_node_t*me, int iport, int index)
{
  int last_port, i, j;
  lx_node_t*n;
  
  insist (fp && m && me && !me->xbar);
  insist (iport >=0 && iport < MI_MY_SIZE);
  
  for (i = 0; i < m->num_hosts; i++)
  {
    n = mi_index2node (m->host_array [i]);
    insist (n && !n->xbar);
    for (j = 0; j < LX_HOST_SIZE; j++)
    {
      if (!lx_get_node (me, iport))
	lx_route_file_write_route (fp, iport, n->index, index, j, 0);
      else if (lx_get_node (n, j))
      {
	lx_node_t *fn;
	fn = lx_follow (me, iport, &lx_host_c (n)->routes [iport] [j], &last_port);
	if (!m->num_xbars && (fn != n || last_port != j)) {
	  lx_route_file_write_route (fp, iport, n->index, index, j, 0);
	} else {
	    insist(fn == n && last_port == j);
	    lx_route_file_write_route (fp, iport, n->index, index, j, &lx_host_c (n)->routes [iport] [j]);
	}
      }
      else
	lx_route_file_write_route (fp, iport, n->index, index, j, 0);
    }
  }
  
  return 1;
  except: return 0;
}

lx_route_table_entry_t *
lx_route_file_read_routes (FILE*fp, int*from, int*num_hosts,
                           int*num_ports, int*num_routes, int*line_count)
{
  lx_route_table_entry_t*entries = 0;
  lx_route_table_entry_t*e;
  lx_route_t route;
  int i, c, r, to, iport, oiport, num_entries, index;
  char line [LX_ROUTE_FILE_MAX_LINE + 1];
  char route_string [LX_ROUTE_FILE_MAX_LINE + 1];

  char*s;
  
  insist (fp && from && num_hosts && num_ports && num_routes && line_count);

  if (!(s = lx_route_file_next_line (fp, line, LX_ROUTE_FILE_MAX_LINE, line_count)))
    return 0;
  
  if (sscanf (s, "h%d %d %d %d\n", from, num_hosts, num_ports, num_routes) != 4 ||
      *num_hosts < 1 || *num_ports < 1 || *num_ports > LX_ROUTE_FILE_MAX_PORTS || *num_routes < 1)
    lx_route_file_error (*line_count, ("bad header"));
  
  num_entries = *num_ports * *num_ports * *num_hosts * *num_routes;
  
  if (!(entries = (lx_route_table_entry_t*) calloc (num_entries, sizeof (*entries) )))
    lx_route_file_error (*line_count, ("malloc failed"));
  
  for (i = 0; i < num_entries; i++)
    entries [i].route.length = (unsigned char) -1;
  
  for (i = 0; i < num_entries; i++)
  {
    if (!(s = lx_route_file_next_line (fp, line, LX_ROUTE_FILE_MAX_LINE, line_count)))
      continue;
    
    route.length = 0;
    r = 1;
    if ((c = sscanf (s, "%d h%d %d %d %s", &iport, &to, &index, &oiport, route_string)) < 4 ||
	index < 0 || index >= *num_routes ||
	to < 0 || to >= *num_hosts || 
	iport < 0 || iport >= *num_ports ||
	oiport < 0 || oiport >= *num_ports ||
	(c == 5 && !(r = lx_route_file_read_route (route_string, &route))))
      lx_route_file_error (*line_count, ("bad route"));
    
    e = lx_route_file_entry (entries, *num_ports, *num_hosts, *num_routes, iport, to, oiport, index);
    if (e->valid)
      lx_route_file_error (*line_count, ("duplicate route"));
    e->valid = (r > 0);
    e->route = route;
  }

  return entries;
  error:
  if (entries) free (entries);
  except: return 0;
}
